const { pool } = require('../config/database');
const { sendNotificationToAll } = require('../services/notificationService');
const bcrypt = require('bcryptjs');

// ## FUNGSI createJadwal (VERSI BARU DENGAN LOGIKA ADAPTIF) ##
exports.createJadwal = async (req, res) => {
    // Mengambil semua input dari admin
    let { 
        tgl_jemput, 
        satuan_pendidikan,
        jumlah_santri, 
        waktu_mulai, 
        waktu_selesai, 
        jumlah_loket, 
        jumlah_sesi,
        jenis_jadwal
    } = req.body;

    // Menangani multi satuan pendidikan
    const satuanPendidikanString = Array.isArray(satuan_pendidikan) 
        ? satuan_pendidikan.join(', ') 
        : satuan_pendidikan;
    
    // Konversi input
    jumlah_santri = parseInt(jumlah_santri, 10) || 0;
    jumlah_loket = parseInt(jumlah_loket, 10) || 1; // Default ke 1 jika kosong/invalid
    jumlah_sesi = parseInt(jumlah_sesi, 10) || 1; // Default ke 1 jika kosong/invalid
    
    // Validasi input
    if (!tgl_jemput || !satuan_pendidikan || !jumlah_santri || !waktu_mulai || !waktu_selesai || !jumlah_loket || !jumlah_sesi || !jenis_jadwal) {
        return res.status(400).json({ message: 'Semua kolom wajib diisi.' });
    }

    const userId = req.user.id;
    const connection = await pool.getConnection();

    try {
        await connection.beginTransaction();
        // Log data mentah
        console.log('Data mentah yang diterima Backend:', req.body);
        console.log('String yang akan disimpan ke database:', satuanPendidikanString);

        // Menghitung kuota berdasarkan jumlah santri
        const kuotaPoinPerSesi = Math.ceil(jumlah_santri / jumlah_sesi);
        const kuotaPrioritasPerSesi = Math.round(kuotaPoinPerSesi * 0.6);

        // Menghitung durasi dari rentang waktu admin
        const [jamMulai, menitMulai] = waktu_mulai.split(':').map(Number);
        const [jamSelesai, menitSelesai] = waktu_selesai.split(':').map(Number);
        const durasiTotalMenit = (jamSelesai * 60 + menitSelesai) - (jamMulai * 60 + menitMulai);

        if (durasiTotalMenit <= 0) {
            // Rollback karena input tidak valid, tidak perlu lanjut
            await connection.rollback();
            return res.status(400).json({ message: 'Rentang waktu tidak valid.' });
        }
        const durasiPerSesiMenit = Math.floor(durasiTotalMenit / jumlah_sesi);

        // Log hasil kalkulasi
        console.log('Hasil Kalkulasi Logika:', {
            kuotaPoinPerSesi,
            kuotaPrioritasPerSesi,
            durasiTotalMenit,
            durasiPerSesiMenit
        });

    // Simpan jadwal utama
    const [resultJadwal] = await connection.query(
        `INSERT INTO jadwal_penjemputan (user_id, satuan_pendidikan, jenis_jadwal, tgl_jemput, waktu_mulai, waktu_selesai, jumlah_sesi, total_santri_dijadwalkan, jumlah_loket) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [userId, satuanPendidikanString, jenis_jadwal, tgl_jemput, waktu_mulai, waktu_selesai, jumlah_sesi, jumlah_santri, jumlah_loket]
    );
    const jadwalId = resultJadwal.insertId;

    // Simpan sesi-sesi
    let waktuMulaiSesiMenit = jamMulai * 60 + menitMulai;
    for (let i = 0; i < jumlah_sesi; i++) {
        const waktuSelesaiSesiMenit = waktuMulaiSesiMenit + durasiPerSesiMenit;
        const formatWaktuMulai = `${String(Math.floor(waktuMulaiSesiMenit / 60)).padStart(2, '0')}:${String(waktuMulaiSesiMenit % 60).padStart(2, '0')}:00`;
        const formatWaktuSelesai = `${String(Math.floor(waktuSelesaiSesiMenit / 60)).padStart(2, '0')}:${String(waktuSelesaiSesiMenit % 60).padStart(2, '0')}:00`;
        
        await connection.query(
            `INSERT INTO sesi_penjemputan (jadwal_id, waktu_mulai, waktu_selesai, kapasitas_poin_total, kapasitas_poin_prioritas) VALUES (?, ?, ?, ?, ?)`, 
            [jadwalId, formatWaktuMulai, formatWaktuSelesai, kuotaPoinPerSesi, kuotaPrioritasPerSesi]
        );
        waktuMulaiSesiMenit = waktuSelesaiSesiMenit;
    }
    await connection.commit();

    //FEEDBACK (SETELAH SEMUA PROSES SUKSES)
    let feedback = { type: 'info', message: 'Jumlah loket yang Anda sediakan sudah pas dengan kebutuhan.' };
    if (durasiPerSesiMenit > 0) {
        const KECEPATAN_PER_LOKET = 0.25;
        const kecepatanDibutuhkan = kuotaPoinPerSesi / durasiPerSesiMenit;
        const loketIdeal = Math.ceil(kecepatanDibutuhkan / KECEPATAN_PER_LOKET);
        if (jumlah_loket > loketIdeal) {
            feedback = { 
                type: 'success', 
                message: `Jadwal ini sangat ideal. Dengan ${jumlah_loket} loket, pelayanan akan berjalan sangat lancar (kebutuhan minimal hanya ${loketIdeal} loket).` 
            };
        } else if (jumlah_loket < loketIdeal) {
            feedback = { 
                type: 'warning', 
                message: `PERINGATAN: Untuk kelancaran, jadwal ini membutuhkan minimal ${loketIdeal} loket. Anda hanya menyediakan ${jumlah_loket} loket. Harap antisipasi antrean.` 
            };
        }
    }
    
    // Kirim notifikasi ke user
    const notificationPayload = {
        title: 'Jadwal Baru Telah Terbit!',
        body: `Jadwal baru untuk satuan pendidikan ${satuanPendidikanString} telah dibuka.`
    };
    sendNotificationToAll(notificationPayload).catch(err => console.error("Gagal mengirim notifikasi:", err));

    // Kirim response ke admin
    res.status(201).json({ 
        message: `Jadwal berhasil dibuat!`,
        feedback: feedback
    });

} catch (error) {
    await connection.rollback();
    // Log error jika terjadi kesalahan
    console.error("Error saat membuat jadwal:", error);
    res.status(500).json({ message: 'Gagal membuat jadwal, terjadi kesalahan pada server.' });
} finally {
    connection.release();
}
};

// FUNGSI updateJadwal
exports.updateJadwal = async (req, res) => {
    const { id: jadwalId } = req.params;
    
    // Mengambil data input
    let { 
        tgl_jemput, 
        satuan_pendidikan,
        jumlah_santri, 
        waktu_mulai, 
        waktu_selesai, 
        jumlah_loket, 
        jumlah_sesi,
        jenis_jadwal
    } = req.body;

    const satuanPendidikanString = Array.isArray(satuan_pendidikan) ? satuan_pendidikan.join(', ') : satuan_pendidikan;

    jumlah_santri = parseInt(jumlah_santri, 10) || 0;
    jumlah_loket = parseInt(jumlah_loket, 10) || 1;
    jumlah_sesi = parseInt(jumlah_sesi, 10) || 1;

    if (!tgl_jemput || !satuan_pendidikan || !jumlah_santri || !waktu_mulai || !waktu_selesai || !jumlah_loket || !jumlah_sesi || !jenis_jadwal) {
        return res.status(400).json({ message: 'Semua kolom wajib diisi.' });
    }

    const connection = await pool.getConnection();
    try {
        await connection.beginTransaction();

        // Cek jadwal
        const [jadwal] = await connection.query('SELECT * FROM jadwal_penjemputan WHERE jadwal_id = ? FOR UPDATE', [jadwalId]);
        if (jadwal.length === 0) {
            await connection.rollback();
            return res.status(404).json({ message: 'Jadwal tidak ditemukan.' });
        }

        // Hapus sesi lama dan update jadwal utama
        await connection.query('DELETE FROM sesi_penjemputan WHERE jadwal_id = ?', [jadwalId]);
        await connection.query(
            `UPDATE jadwal_penjemputan SET satuan_pendidikan = ?, jenis_jadwal = ?, tgl_jemput = ?, waktu_mulai = ?, waktu_selesai = ?, jumlah_sesi = ?, total_santri_dijadwalkan = ?, jumlah_loket = ? WHERE jadwal_id = ?`,
            [satuanPendidikanString, jenis_jadwal, tgl_jemput, waktu_mulai, waktu_selesai, jumlah_sesi, jumlah_santri, jumlah_loket, jadwalId]
        );
        const kuotaPoinPerSesi = Math.ceil(jumlah_santri / jumlah_sesi);
        const kuotaPrioritasPerSesi = Math.floor(kuotaPoinPerSesi * 0.6);

        const [jamMulai, menitMulai] = waktu_mulai.split(':').map(Number);
        const [jamSelesai, menitSelesai] = waktu_selesai.split(':').map(Number);
        const durasiTotalMenit = (jamSelesai * 60 + menitSelesai) - (jamMulai * 60 + menitMulai);
        const durasiPerSesiMenit = Math.floor(durasiTotalMenit / jumlah_sesi);

        // Buat kembali sesi-sesi baru dengan kuota yang benar
        let waktuMulaiSesiMenit = jamMulai * 60 + menitMulai;
        for (let i = 0; i < jumlah_sesi; i++) {
            const waktuSelesaiSesiMenit = waktuMulaiSesiMenit + durasiPerSesiMenit;
            const formatWaktuMulai = `${String(Math.floor(waktuMulaiSesiMenit / 60)).padStart(2, '0')}:${String(waktuMulaiSesiMenit % 60).padStart(2, '0')}:00`;
            const formatWaktuSelesai = `${String(Math.floor(waktuSelesaiSesiMenit / 60)).padStart(2, '0')}:${String(waktuSelesaiSesiMenit % 60).padStart(2, '0')}:00`;
            
            await connection.query(
                `INSERT INTO sesi_penjemputan (jadwal_id, waktu_mulai, waktu_selesai, kapasitas_poin_total, kapasitas_poin_prioritas) VALUES (?, ?, ?, ?, ?)`, 
                [jadwalId, formatWaktuMulai, formatWaktuSelesai, kuotaPoinPerSesi, kuotaPrioritasPerSesi]
            );
            waktuMulaiSesiMenit = waktuSelesaiSesiMenit;
        }

        await connection.commit();

        // FEEDBACK UNTUK ADMIN
        let feedback = { type: 'info', message: 'Jumlah loket yang Anda sediakan sudah pas dengan kebutuhan.' };
        if (durasiPerSesiMenit > 0) {
            const KECEPATAN_PER_LOKET = 0.25;
            const kecepatanDibutuhkan = kuotaPoinPerSesi / durasiPerSesiMenit;
            const loketIdeal = Math.ceil(kecepatanDibutuhkan / KECEPATAN_PER_LOKET);

            if (jumlah_loket > loketIdeal) {
                feedback = { type: 'success', message: `Jadwal ini sangat ideal. Dengan ${jumlah_loket} loket, pelayanan akan berjalan sangat lancar (kebutuhan minimal hanya ${loketIdeal} loket).` };
            } else if (jumlah_loket < loketIdeal) {
                feedback = { type: 'warning', message: `PERINGATAN: Untuk kelancaran, jadwal ini membutuhkan minimal ${loketIdeal} loket. Anda hanya menyediakan ${jumlah_loket} loket. Harap antisipasi antrean.` };
            }
        }

        res.status(200).json({ 
            message: `Jadwal dan sesinya berhasil diperbarui!`,
            feedback: feedback 
        });

    } catch (error) {
        await connection.rollback();
        console.error("Error saat memperbarui jadwal:", error);
        res.status(500).json({ message: 'Gagal memperbarui jadwal.', error: error.message });
    } finally {
        connection.release();
    }
};

exports.getAllOrangTua = async (req, res) => {
    try {
        const [users] = await pool.query("SELECT id as user_id, nama_wali, username, email, no_telp FROM users WHERE role = 'orang_tua'");
        res.status(200).json(users);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

exports.deleteJadwal = async (req, res) => {
    try {
        const { id } = req.params;
        const [result] = await pool.query('DELETE FROM jadwal_penjemputan WHERE jadwal_id = ?', [id]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ message: 'Jadwal tidak ditemukan.' });
        }
        res.status(200).json({ message: `Jadwal ID ${id} berhasil dihapus.` });
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

exports.getDashboardData = async (req, res) => {
    try {
        const [totalDijadwalkanResult] = await pool.query(`SELECT SUM(total_santri_dijadwalkan) AS grand_total FROM jadwal_penjemputan WHERE tgl_jemput >= CURDATE()`);
        const total_santri_dijadwalkan = totalDijadwalkanResult[0].grand_total || 0;
        const [totalTerdaftarResult] = await pool.query(`SELECT COUNT(dp.detail_id) AS grand_total FROM detail_penjemputan dp JOIN pilihan_sesi ps ON dp.pilihan_id = ps.pilihan_id JOIN sesi_penjemputan ss ON ps.sesi_id = ss.sesi_id JOIN jadwal_penjemputan jp ON ss.jadwal_id = jp.jadwal_id WHERE jp.tgl_jemput >= CURDATE()`);
        const total_antrian_terambil = totalTerdaftarResult[0].grand_total || 0;
        const [jadwalPenjemputan] = await pool.query(`SELECT * FROM jadwal_penjemputan WHERE jenis_jadwal = 'penjemputan' ORDER BY created_at DESC LIMIT 4`);
        const [jadwalKembali] = await pool.query(`SELECT * FROM jadwal_penjemputan WHERE jenis_jadwal = 'kembali' ORDER BY created_at DESC LIMIT 4`);
        const jadwalList = [...jadwalPenjemputan, ...jadwalKembali];
        if (jadwalList.length === 0) {
            return res.status(200).json({ total_santri_dijadwalkan, total_antrian_terambil, jadwal_terbaru: [] });
        }
        const jadwalIds = jadwalList.map(j => j.jadwal_id);
        const [sesiList] = await pool.query(`SELECT * FROM sesi_penjemputan WHERE jadwal_id IN (?) ORDER BY waktu_mulai ASC`, [jadwalIds]);
        const [registeredCount] = await pool.query(`SELECT ss.jadwal_id, COUNT(dp.detail_id) AS jumlah_terdaftar FROM detail_penjemputan dp JOIN pilihan_sesi ps ON dp.pilihan_id = ps.pilihan_id JOIN sesi_penjemputan ss ON ps.sesi_id = ss.sesi_id WHERE ss.jadwal_id IN (?) GROUP BY ss.jadwal_id`, [jadwalIds]);
        const jadwal_terbaru = jadwalList.map(jadwal => {

    // Komentar: Kita menambahkan 'index' ke dalam fungsi .map() untuk mendapatkan nomor urut.
    const sesiUntukJadwalIni = sesiList
        .filter(s => s.jadwal_id === jadwal.jadwal_id)
        .map((s, index) => { 
            const isPenuh = (s.poin_terpakai || 0) >= s.kapasitas_poin_total;
            const status = isPenuh ? 'Penuh' : 'Tersedia';
            
            // Menambahkan nomor_sesi ke setiap objek sesi
            return { ...s, nomor_sesi: index + 1, status_ketersediaan: status }; 
        });

    const pendaftar = registeredCount.find(c => c.jadwal_id === jadwal.jadwal_id);
    return { ...jadwal, jumlah_santri_terdaftar: pendaftar ? pendaftar.jumlah_terdaftar : 0, sesi: sesiUntukJadwalIni };
});
        res.status(200).json({
            total_santri_dijadwalkan,
            total_antrian_terambil,
            jadwal_terbaru
        });
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

exports.getAllJadwalForManagement = async (req, res) => {
    try {
        const [jadwalList] = await pool.query(
            `SELECT * FROM jadwal_penjemputan ORDER BY tgl_jemput DESC, created_at DESC`
        );
        res.status(200).json(jadwalList);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

// ## FUNGSI getSesiDetail
exports.getSesiDetail = async (req, res) => {
    try {
        const { sesi_id } = req.params;
        const query = `
            SELECT 
                u.nama_wali,
                ps.jenis_kendaraan,
                ps.jalur_digunakan,
                ps.jumlah_santri,
                (SELECT GROUP_CONCAT(s.nama_santri SEPARATOR ', ') 
                 FROM detail_penjemputan dp 
                 JOIN santri s ON dp.santri_id = s.santri_id 
                 WHERE dp.pilihan_id = ps.pilihan_id) AS daftar_santri
            FROM pilihan_sesi ps
            JOIN users u ON ps.user_id = u.id
            WHERE ps.sesi_id = ?
            ORDER BY ps.tgl_pilihan ASC;
        `;
        const [results] = await pool.query(query, [sesi_id]);
        res.status(200).json(results);
    } catch (error) {
        console.error('Gagal mengambil detail sesi:', error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

exports.changePassword = async (req, res) => {
    const { currentPassword, newPassword, confirmPassword } = req.body;
    const adminId = req.user.id; // Diambil dari token JWT middleware

    // Validasi input dasar
    if (!currentPassword || !newPassword || !confirmPassword) {
        return res.status(400).json({ message: 'Semua kolom wajib diisi.' });
    }
    if (newPassword !== confirmPassword) {
        return res.status(400).json({ message: 'Password baru dan konfirmasi tidak cocok.' });
    }
    if (newPassword.length < 8) {
        return res.status(400).json({ message: 'Password baru minimal harus 8 karakter.' });
    }

    try {
        // Ambil data admin dari database
        const [rows] = await pool.query('SELECT * FROM users WHERE id = ? AND role = "admin"', [adminId]);
        if (rows.length === 0) {
            // Ini seharusnya tidak terjadi jika middleware bekerja, tapi sebagai pengaman
            return res.status(404).json({ message: 'Akun admin tidak ditemukan.' });
        }
        const admin = rows[0];

        // Verifikasi password saat ini
        const isMatch = await bcrypt.compare(currentPassword, admin.password);
        if (!isMatch) {
            return res.status(400).json({ message: 'Password saat ini yang Anda masukkan salah.' });
        }

        // Hash password baru
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(newPassword, salt);

        // Update password baru ke database
        await pool.query('UPDATE users SET password = ? WHERE id = ?', [hashedPassword, adminId]);

        res.status(200).json({ message: 'Password Anda telah berhasil diperbarui.' });

    } catch (error) {
        console.error('Gagal saat proses ganti password admin:', error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server, silakan coba lagi.' });
    }
};

// ## FUNGSI UNTUK HALAMAN DETAIL JADWAL ##
exports.getJadwalDetail = async (req, res) => {
    try {
        const { jadwal_id } = req.params;

        // Ambil data utama jadwal
        const [jadwalResult] = await pool.query('SELECT * FROM jadwal_penjemputan WHERE jadwal_id = ?', [jadwal_id]);
        if (jadwalResult.length === 0) {
            return res.status(404).json({ message: 'Jadwal tidak ditemukan.' });
        }
        const jadwalData = jadwalResult[0];

        // Ambil semua sesi yang terkait dengan jadwal ini
        const [sesiList] = await pool.query('SELECT * FROM sesi_penjemputan WHERE jadwal_id = ? ORDER BY waktu_mulai ASC', [jadwal_id]);
        
        // Tambahkan nomor sesi dan hitung status untuk setiap sesi
        const sesiDetail = sesiList.map((sesi, index) => {
            const poinTerpakai = sesi.poin_terpakai || 0;
            const isPenuh = poinTerpakai >= sesi.kapasitas_poin_total;
            const status = isPenuh ? 'Penuh' : 'Tersedia';
            
            return {
                ...sesi,
                nomor_sesi: index + 1,
                status_ketersediaan: status,
                kapasitas_santri_total: Math.round(sesi.kapasitas_poin_total), // 1 poin = 1 santri untuk display
                santri_terdaftar: Math.round(poinTerpakai), 
                reguler_terpakai: Math.round(sesi.poin_reguler_terpakai || 0),
                kapasitas_reguler: sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas,
                prioritas_terpakai: Math.round(sesi.poin_prioritas_terpakai || 0),
                kapasitas_prioritas: sesi.kapasitas_poin_prioritas
            };
        });

        // Gabungkan data jadwal dan sesi, lalu kirimkan
        const responseData = {
            ...jadwalData,
            sesi: sesiDetail
        };

        res.status(200).json(responseData);

    } catch (error) {
        console.error('Gagal mengambil detail jadwal:', error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

//FUNGSI HALAMAN DETAIL PENDAFTAR
exports.getSesiPendaftarDetail = async (req, res) => {
    try {
        const { sesi_id } = req.params;
        
        // Query dengan logika 'CASE' untuk menentukan hubungan yang benar
        const query = `
            SELECT 
                -- 1. Mengambil nama dari user yang melakukan pemesanan (penjemput)
                u.nama_wali AS nama_penjemput,
                
                -- 2. Mengambil nama santri yang dijemput
                s.nama_santri,
                
                -- 3. LOGIKA KUNCI: Menentukan hubungan secara dinamis
                CASE
                    -- Jika ID penjemput (ps.user_id) SAMA DENGAN ID wali asli santri (s.user_id)
                    WHEN ps.user_id = s.user_id THEN s.hubungan_wali
                    -- Jika berbeda, maka ini adalah santri titipan
                    ELSE 'Santri Titipan'
                END AS hubungan,
                
                -- 4. Data lainnya
                ps.jenis_kendaraan,
                ps.jalur_digunakan
            FROM pilihan_sesi ps
            -- Join ke tabel 'users' untuk mendapatkan NAMA PENJEMPUT
            JOIN users u ON ps.user_id = u.id
            JOIN detail_penjemputan dp ON ps.pilihan_id = dp.pilihan_id
            -- Join ke tabel 'santri' untuk mendapatkan info santri dan ID wali aslinya
            JOIN santri s ON dp.santri_id = s.santri_id
            WHERE ps.sesi_id = ?
            -- Mengurutkan berdasarkan nama penjemput untuk mengelompokkan secara visual
            ORDER BY ps.tgl_pilihan ASC;
        `;
        
        const [results] = await pool.query(query, [sesi_id]);
        res.status(200).json(results);

    } catch (error) {
        console.error('Gagal mengambil detail pendaftar sesi:', error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};