const { pool } = require('../config/database');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const dotenv = require('dotenv');
const crypto = require('crypto');
const sendEmail = require('../config/email');

dotenv.config();

if (!process.env.JWT_SECRET) {
    console.error('FATAL ERROR: JWT_SECRET tidak terdefinisi di dalam file .env');
    process.exit(1);
}

// FUNGSI REGISTRASI
exports.register = async (req, res) => {
    const { username, email, password } = req.body;

    if (!username || !email || !password) {
        return res.status(400).json({ message: 'Username, email, dan password wajib diisi!' });
    }

    try {
        // Cek duplikat di tabel users baru
        const [existingUser] = await pool.query(
            'SELECT * FROM users WHERE username = ? OR email = ?',
            [username, email]
        );

        if (existingUser.length > 0) {
            return res.status(409).json({ message: 'Username atau email sudah terdaftar.' });
        }

        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(password, salt);

        // Simpan ke tabel users dengan role 'orang_tua' secara eksplisit
        await pool.query(
            "INSERT INTO users (username, email, password, role) VALUES (?, ?, ?, 'orang_tua')",
            [username, email, hashedPassword]
        );

        res.status(201).json({ message: 'Registrasi berhasil! Silakan login.' });

    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

// FUNGSI LOGIN
exports.login = async (req, res) => {
    const { loginIdentifier, password } = req.body;

    if (!loginIdentifier || !password) {
        return res.status(400).json({ message: 'Username/Email dan password wajib diisi.' });
    }

    const connection = await pool.getConnection();
    try {
        const [users] = await connection.query(
            'SELECT * FROM users WHERE username = ? OR email = ?', 
            [loginIdentifier, loginIdentifier]
        );

        if (users.length === 0) {
            return res.status(401).json({ message: 'Kombinasi Username/Email dan Password salah.' });
        }

        const user = users[0];
        const isMatch = await bcrypt.compare(password, user.password);
        if (!isMatch) {
            return res.status(401).json({ message: 'Kombinasi Username/Email dan Password salah.' });
        }

        const payload = {
            id: user.id, 
            username: user.username,
            role: user.role // Ambil role langsung dari database
        };
        
        const expiresIn = user.role === 'admin' ? '8h' : '4h';

        const token = jwt.sign(payload, process.env.JWT_SECRET, {
            expiresIn: expiresIn 
        });

        res.status(200).json({
            message: 'Login berhasil!',
            token: token
        });

    } catch (error) {
        console.error("Error saat login:", error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    } finally {
        if (connection) {
            connection.release();
        }
    }
};

// FUNGSI LUPA & RESET PASSWORD
exports.forgotPassword = async (req, res) => {
    const { email } = req.body;
    if (!email) {
        return res.status(400).json({ message: 'Email wajib diisi.' });
    }

    try {
        const [user] = await pool.query("SELECT * FROM users WHERE email = ? AND role = 'orang_tua'", [email]);
        if (user.length === 0) {
            return res.status(200).json({ message: 'Jika email terdaftar, instruksi reset akan dikirim.' });
        }

        const resetToken = crypto.randomBytes(20).toString('hex');
        const passwordResetToken = crypto.createHash('sha256').update(resetToken).digest('hex');
        const passwordResetExpires = new Date(Date.now() + 10 * 60 * 1000);

        await pool.query(
            'UPDATE users SET passwordResetToken = ?, passwordResetExpires = ? WHERE email = ?',
            [passwordResetToken, passwordResetExpires, email]
        );

        const resetURL = `https://penjemputan-ppum.site/reset-password.html?token=${resetToken}`;
        const message = `...`; // Isi pesan email

        try {
            await sendEmail({ email: user[0].email, subject: 'Link Reset Password', message });
            res.status(200).json({ message: 'Link reset password telah dikirim ke email Anda.' });
        } catch (err) {
            console.error('ERROR SAAT MENGIRIM EMAIL:', err);
            await pool.query('UPDATE users SET passwordResetToken = NULL, passwordResetExpires = NULL WHERE email = ?', [email]);
            res.status(500).json({ message: 'Gagal mengirim email. Silakan coba lagi nanti.' });
        }

    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

exports.resetPassword = async (req, res) => {
    const { token } = req.params;
    const { password, confirmPassword } = req.body;

    if (!password || !confirmPassword || password !== confirmPassword) {
        return res.status(400).json({ message: 'Password baru dan konfirmasi password tidak cocok.' });
    }

    const hashedToken = crypto.createHash('sha256').update(token).digest('hex');

    try {
        const [user] = await pool.query(
            'SELECT * FROM users WHERE passwordResetToken = ? AND passwordResetExpires > NOW()',
            [hashedToken]
        );

        if (user.length === 0) {
            return res.status(400).json({ message: 'Token tidak valid atau sudah kedaluwarsa.' });
        }

        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(password, salt);

        await pool.query(
            'UPDATE users SET password = ?, passwordResetToken = NULL, passwordResetExpires = NULL WHERE id = ?',
            [hashedPassword, user[0].id]
        );

        res.status(200).json({ message: 'Password berhasil direset. Silakan login kembali.' });

    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};