const { pool } = require('../config/database');
const dayjs = require('dayjs');

// Logika Greedy dengan Parameter Waktu
const determineJalur = (sesi, biayaPoin, jenis_kendaraan, tgl_jemput) => {
    const isPrioritas = (jenis_kendaraan === 'mobil' && biayaPoin >= 2) || jenis_kendaraan === 'motor';
    
    // Aturan fleksibel berlaku jika hari ini adalah H-1 atau hari H.
    const BATAS_HARI_FLEKSIBEL = 1; // 1 hari sebelum
    const hariIni = dayjs().startOf('day');
    const hariJadwal = dayjs(tgl_jemput).startOf('day');
    const selisihHari = hariJadwal.diff(hariIni, 'day');
    
    const isFaseFleksibel = selisihHari <= BATAS_HARI_FLEKSIBEL;

    // LOGIKA UTAMA BERDASARKAN WAKTU

    if (isFaseFleksibel) {
        // FASE FLEKSIBEL (H-1 atau Hari-H)
        // Menggunakan logika yang memaksimalkan utilitas. Reguler boleh "meminjam" sisa kuota prioritas.
        if (isPrioritas && (sesi.poin_prioritas_terpakai + biayaPoin) <= sesi.kapasitas_poin_prioritas) {
            return 'prioritas';
        } else {
            const kapasitasReguler = sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas;
            if ((sesi.poin_reguler_terpakai + biayaPoin) <= kapasitasReguler) {
                return 'reguler';
            } else if ((sesi.poin_terpakai + biayaPoin) <= sesi.kapasitas_poin_total) {
                return 'reguler';
            }
        }
    } else {
        // FASE KETAT (Lebih dari H-1)
        // Logika di sini lebih kaku untuk menjamin hak user prioritas. Reguler TIDAK boleh meminjam kuota prioritas.
        if (isPrioritas) {
            // User prioritas mencoba jalur prioritas, jika gagal atau full boleh ke reguler.
            if ((sesi.poin_prioritas_terpakai + biayaPoin) <= sesi.kapasitas_poin_prioritas) {
                return 'prioritas';
            } else {
                const kapasitasReguler = sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas;
                if ((sesi.poin_reguler_terpakai + biayaPoin) <= kapasitasReguler) {
                    return 'reguler';
                }
            }
        } else {
            // User reguler HANYA boleh mencoba jalur reguler.
            const kapasitasReguler = sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas;
            if ((sesi.poin_reguler_terpakai + biayaPoin) <= kapasitasReguler) {
                return 'reguler';
            }
        }
    }
    return null; 
};

const getAllJadwal = async (req, res) => {
    try {
        const user_id = req.user.id;
        const [jenjangSantriSendiri] = await pool.query(
            `SELECT DISTINCT satuan_pendidikan FROM santri WHERE user_id = ?`, [user_id]
        );
        const [jenjangSantriTitipan] = await pool.query(
            `SELECT DISTINCT s.satuan_pendidikan 
             FROM izin_penjemputan i JOIN santri s ON i.santri_id = s.santri_id 
             WHERE i.user_id_penjemput = ? AND i.status = 'menunggu'`, 
            [user_id]
        );
        
        const jenjangRelevan = [...new Set([
            ...jenjangSantriSendiri.map(j => j.satuan_pendidikan),
            ...jenjangSantriTitipan.map(j => j.satuan_pendidikan)
        ])];

        const [penjemputanList] = await pool.query(
            `SELECT jadwal_id, satuan_pendidikan, tgl_jemput, waktu_mulai, waktu_selesai, jenis_jadwal 
             FROM jadwal_penjemputan 
             WHERE tgl_jemput >= CURDATE() AND jenis_jadwal = 'penjemputan'
             ORDER BY tgl_jemput ASC, waktu_mulai ASC LIMIT 4`
        );

        const [kembaliList] = await pool.query(
            `SELECT jadwal_id, satuan_pendidikan, tgl_jemput, waktu_mulai, waktu_selesai, jenis_jadwal
             FROM jadwal_penjemputan 
             WHERE tgl_jemput >= CURDATE() AND jenis_jadwal = 'kembali'
             ORDER BY tgl_jemput ASC, waktu_mulai ASC LIMIT 4`
        );
        
        const tandaiRelevansi = (jadwal) => {
            const jenjangJadwal = jadwal.satuan_pendidikan.split(', ');
            const isRelevan = jenjangJadwal.some(jj => jenjangRelevan.includes(jj));
            return { ...jadwal, isRelevan: isRelevan };
        };

        const penjemputanRelevan = penjemputanList.map(tandaiRelevansi);
        const kembaliRelevan = kembaliList.map(tandaiRelevansi);
        
        res.status(200).json({
            penjemputan: penjemputanRelevan,
            kembali: kembaliRelevan
        });

    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

const getJadwalById = async (req, res) => {
    try {
        const { id } = req.params;
        const [jadwal] = await pool.query('SELECT * FROM jadwal_penjemputan WHERE jadwal_id = ?', [id]);
        if (jadwal.length === 0) {
            return res.status(404).json({ message: 'Jadwal tidak ditemukan.' });
        }
        
        const [sesiList] = await pool.query('SELECT * FROM sesi_penjemputan WHERE jadwal_id = ? ORDER BY waktu_mulai ASC', [id]);
        
        const sesiFinal = sesiList.map((sesi, index) => {
            const isPenuh = (sesi.poin_terpakai || 0) >= sesi.kapasitas_poin_total;
            const status = isPenuh ? 'Penuh' : 'Tersedia';
            const kapasitasReguler = sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas;
            const poinRegulerTerpakai = sesi.poin_reguler_terpakai || 0;

            return {
                sesi_id: sesi.sesi_id,
                waktu_mulai: sesi.waktu_mulai,
                waktu_selesai: sesi.waktu_selesai,
                nomor_sesi: index + 1,
                status_ketersediaan: status,
                kapasitas_poin_total: sesi.kapasitas_poin_total,
                poin_terpakai: sesi.poin_terpakai || 0,
                sisa_poin_total: sesi.kapasitas_poin_total - (sesi.poin_terpakai || 0),
                sisa_poin_prioritas: sesi.kapasitas_poin_prioritas - (sesi.poin_prioritas_terpakai || 0),
                sisa_poin_reguler: kapasitasReguler - poinRegulerTerpakai
            };
        });

        const responseData = { ...jadwal[0], sesi: sesiFinal };
        res.status(200).json(responseData);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

const pilihSesi = async (req, res) => {
    const { sesi_id, santri_ids, jenis_kendaraan } = req.body;
    const user_id = req.user.id;

    if (!sesi_id || !santri_ids || !jenis_kendaraan || !Array.isArray(santri_ids) || santri_ids.length === 0) {
        return res.status(400).json({ message: 'Informasi sesi, daftar santri, dan jenis kendaraan wajib diisi.' });
    }

    const connection = await pool.getConnection();
    try {
        await connection.beginTransaction();

        // PERUBAHAN UTAMA 2: Mengambil tgl_jemput untuk logika waktu
        const [sesiArr] = await connection.query(
            `SELECT s.*, j.tgl_jemput 
             FROM sesi_penjemputan s 
             JOIN jadwal_penjemputan j ON s.jadwal_id = j.jadwal_id 
             WHERE s.sesi_id = ? FOR UPDATE`, 
            [sesi_id]
        );
        
        if (sesiArr.length === 0) {
            await connection.rollback(); 
            return res.status(404).json({ message: 'Sesi tidak ditemukan.' });
        }
        const sesi = sesiArr[0];

        // Memanggil fungsi determineJalur dengan parameter tambahan 'sesi.tgl_jemput'
        const biayaPoin = santri_ids.length;
        const jalurDigunakan = determineJalur(sesi, biayaPoin, jenis_kendaraan, sesi.tgl_jemput);

        // Logika ini memberikan pesan yang lebih jelas kepada user jika mereka ditolak karena aturan waktu.
        if (!jalurDigunakan) {
            await connection.rollback();
            const hariIni = dayjs().startOf('day');
            const hariJadwal = dayjs(sesi.tgl_jemput).startOf('day');
            const selisihHari = hariJadwal.diff(hariIni, 'day');
            const kapasitasReguler = sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas;

            if (selisihHari > 1 && (sesi.poin_reguler_terpakai >= kapasitasReguler) && (sesi.poin_terpakai < sesi.kapasitas_poin_total)) {
                // Jika gagal karena kuota reguler penuh TAPI total masih ada, dan masih jauh hari
                const tanggalBuka = hariJadwal.subtract(1, 'day').format('D MMMM YYYY');
                return res.status(409).json({ message: `Kuota reguler penuh. Sisa kuota akan dibuka untuk umum pada ${tanggalBuka}.` });
            } else {
                // Pesan error umum jika benar-benar penuh
                return res.status(409).json({ message: 'Kapasitas untuk sesi ini sudah penuh. Silakan pilih sesi lain.' });
            }
        }
        
        const [pilihanResult] = await connection.query(
            'INSERT INTO pilihan_sesi (user_id, sesi_id, jumlah_santri, jenis_kendaraan, jalur_digunakan) VALUES (?, ?, ?, ?, ?)', 
            [user_id, sesi_id, biayaPoin, jenis_kendaraan, jalurDigunakan]
        );
        const pilihanId = pilihanResult.insertId;

        const santriTitipanYangDigunakan = [];
        for (const santriId of santri_ids) {
            const [santriOwner] = await connection.query('SELECT user_id FROM santri WHERE santri_id = ?', [santriId]);
            let isValid = false;
            if (santriOwner.length > 0 && santriOwner[0].user_id === user_id) {
                isValid = true;
            } else {
                const [isAuthorized] = await connection.query('SELECT izin_id FROM izin_penjemputan WHERE santri_id = ? AND user_id_penjemput = ? AND jadwal_id = ? AND status = "menunggu"', [santriId, user_id, sesi.jadwal_id]);
                if (isAuthorized.length > 0) {
                    isValid = true;
                    santriTitipanYangDigunakan.push(santriId);
                }
            }
            if (!isValid) {
                throw new Error(`Anda tidak memiliki izin untuk menjemput santri dengan ID ${santriId}.`);
            }
            await connection.query('INSERT INTO detail_penjemputan (pilihan_id, santri_id) VALUES (?, ?)', [pilihanId, santriId]);
        }
        if (santriTitipanYangDigunakan.length > 0) {
            await connection.query(
                `UPDATE izin_penjemputan SET status = 'digunakan' WHERE user_id_penjemput = ? AND jadwal_id = ? AND santri_id IN (?)`,
                [user_id, sesi.jadwal_id, santriTitipanYangDigunakan]
            );
        }
        
        if (jalurDigunakan === 'prioritas') {
            await connection.query(
                `UPDATE sesi_penjemputan SET poin_terpakai = poin_terpakai + ?, poin_prioritas_terpakai = poin_prioritas_terpakai + ? WHERE sesi_id = ?`,
                [biayaPoin, biayaPoin, sesi_id]
            );
        } else {
            await connection.query(
                `UPDATE sesi_penjemputan SET poin_terpakai = poin_terpakai + ?, poin_reguler_terpakai = poin_reguler_terpakai + ? WHERE sesi_id = ?`,
                [biayaPoin, biayaPoin, sesi_id]
            );
        }

        await connection.query(
            `UPDATE jadwal_penjemputan SET total_santri_terdaftar = total_santri_terdaftar + ? WHERE jadwal_id = ?`,
            [biayaPoin, sesi.jadwal_id]
        );

        await connection.commit();
        res.status(201).json({ message: `${santri_ids.length} santri berhasil didaftarkan untuk sesi ini!` });

    } catch (error) {
        await connection.rollback();
        console.error(error);
        res.status(500).json({ message: 'Gagal memilih sesi.', error: error.message });
    } finally {
        connection.release();
    }
};

const createIzin = async (req, res) => {
    const { jadwal_id, santri_id, username_penjemput } = req.body;
    const user_id_wali_sah = req.user.id; 

    try {
        const connection = await pool.getConnection();
        await connection.beginTransaction();

        const [santriOwner] = await connection.query('SELECT * FROM santri WHERE santri_id = ? AND user_id = ?', [santri_id, user_id_wali_sah]);
        if (santriOwner.length === 0) {
            await connection.rollback();
            return res.status(403).json({ message: 'Akses ditolak: Anda bukan wali sah dari santri ini.' });
        }

        const [penjemput] = await connection.query('SELECT user_id FROM orang_tua WHERE username = ?', [username_penjemput]);
        if (penjemput.length === 0) {
            await connection.rollback();
            return res.status(404).json({ message: `User dengan username '${username_penjemput}' tidak ditemukan.` });
        }
        const user_id_penjemput = penjemput[0].user_id;

        await connection.query(
            `INSERT INTO izin_penjemputan (jadwal_id, santri_id, user_id_wali_sah, user_id_penjemput) VALUES (?, ?, ?, ?)`,
            [jadwal_id, santri_id, user_id_wali_sah, user_id_penjemput]
        );

        await connection.commit();
        res.status(201).json({ message: `Izin berhasil diberikan kepada ${username_penjemput}.` });

    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

// EKSPOR SEMUA FUNGSI DALAM SATU OBJEK
module.exports = {
    determineJalur,
    getAllJadwal,
    getJadwalById,
    pilihSesi,
    createIzin
};