const { determineJalur } = require('./jadwalController.js');
const dayjs = require('dayjs');

// Grup utama untuk semua tes algoritma determineJalur
describe('Unit Testing for determineJalur Algorithm (Time-based)', () => {
    
    // ==========================================================
    // PERSIAPAN DATA & WAKTU UNTUK SEMUA TES
    // ==========================================================
    const tanggalJadwal = '2025-10-30'; // Tanggal penjemputan fiktif
    
    // memalsukan "hari ini" untuk menguji kedua fase
    const H_MINUS_2 = '2025-10-28'; // memicu Fase Ketat
    const H_MINUS_1 = '2025-10-29'; // memicu Fase Fleksibel

    // Data dasar sesi yang akan dimodifikasi di setiap tes
    const sesiDasar = {
        kapasitas_poin_total: 25,
        kapasitas_poin_prioritas: 15, // 60%
        poin_terpakai: 0,
        poin_prioritas_terpakai: 0,
        poin_reguler_terpakai: 0
    };

    // PENGUJIAN FASE KETAT (Pendaftaran dilakukan > H-1)
    describe('Fase Ketat (Pendaftaran > H-1)', () => {
        
        beforeAll(() => {
            // Mengatur "hari ini" menjadi H-2 untuk semua tes di dalam blok describe ini
            jest.useFakeTimers().setSystemTime(new Date(H_MINUS_2));
        });

        afterAll(() => {
            // Mengembalikan waktu ke kondisi normal setelah selesai
            jest.useRealTimers();
        });

        test('✅ [Ketat] User prioritas harus dapat jalur prioritas jika kuota tersedia', () => {
            const mockSesi = { ...sesiDasar, poin_prioritas_terpakai: 10 };
            const result = determineJalur(mockSesi, 2, 'mobil', tanggalJadwal);
            expect(result).toBe('prioritas');
        });

        test('✅ [Ketat] User prioritas harus dialihkan (fallback) ke reguler jika kuota prioritas penuh', () => {
            const mockSesi = { ...sesiDasar, poin_prioritas_terpakai: 15, poin_reguler_terpakai: 5 };
            const result = determineJalur(mockSesi, 2, 'mobil', tanggalJadwal);
            expect(result).toBe('reguler');
        });

        test('✅ [Ketat] User reguler harus dapat jalur reguler jika kuota tersedia', () => {
            const mockSesi = { ...sesiDasar, poin_reguler_terpakai: 5 };
            const result = determineJalur(mockSesi, 1, 'mobil', tanggalJadwal);
            expect(result).toBe('reguler');
        });

        test('🛑 [Ketat] User reguler HARUS DITOLAK jika kuota reguler penuh, meskipun prioritas masih ada', () => {
            const kapasitasReguler = sesiDasar.kapasitas_poin_total - sesiDasar.kapasitas_poin_prioritas; // 10
            const mockSesi = { ...sesiDasar, poin_prioritas_terpakai: 5, poin_reguler_terpakai: kapasitasReguler };
            const result = determineJalur(mockSesi, 1, 'mobil', tanggalJadwal);
            expect(result).toBeNull();
        });
    });

    // PENGUJIAN FASE FLEKSIBEL (Pendaftaran dilakukan pada H-1 atau Hari-H)
    describe('Fase Fleksibel (Pendaftaran <= H-1)', () => {

        beforeAll(() => {
            // Mengatur "hari ini" menjadi H-1 untuk semua tes di dalam blok describe ini
            jest.useFakeTimers().setSystemTime(new Date(H_MINUS_1));
        });

        afterAll(() => {
            jest.useRealTimers();
        });

        test('✅ [Fleksibel] User prioritas harus dapat jalur prioritas jika kuota tersedia', () => {
            const mockSesi = { ...sesiDasar, poin_prioritas_terpakai: 10 };
            const result = determineJalur(mockSesi, 2, 'mobil', tanggalJadwal);
            expect(result).toBe('prioritas');
        });

        test('✅ [Fleksibel] User reguler harus dapat jalur reguler jika kuota tersedia', () => {
            const mockSesi = { ...sesiDasar, poin_reguler_terpakai: 5 };
            const result = determineJalur(mockSesi, 1, 'mobil', tanggalJadwal);
            expect(result).toBe('reguler');
        });

        test('🌟 [Fleksibel] User reguler HARUS BISA mengambil sisa kuota prioritas jika kuota reguler penuh', () => {
            const kapasitasReguler = sesiDasar.kapasitas_poin_total - sesiDasar.kapasitas_poin_prioritas; // 10
            const mockSesi = { 
                ...sesiDasar, 
                poin_prioritas_terpakai: 5, // Prioritas masih sisa banyak
                poin_reguler_terpakai: kapasitasReguler, // Reguler sudah penuh
                poin_terpakai: 5 + kapasitasReguler
            };
            const result = determineJalur(mockSesi, 1, 'mobil', tanggalJadwal);
            expect(result).toBe('reguler');
        });

        test('🛑 [Fleksibel] User manapun harus ditolak jika total kapasitas sesi sudah penuh', () => {
            const mockSesi = { 
                kapasitas_poin_total: 25,
                kapasitas_poin_prioritas: 15,
                poin_terpakai: 25, // Total kapasitas sudah habis
                poin_prioritas_terpakai: 15, // Asumsikan kuota prioritas juga habis
                poin_reguler_terpakai: 10  // Asumsikan kuota reguler juga habis
            };

            const result = determineJalur(mockSesi, 1, 'mobil', tanggalJadwal);
            expect(result).toBeNull();
        });
    });
});