const { pool } = require('../config/database');
const bcrypt = require('bcryptjs');

// FUNGSI Mendapatkan Daftar Santri Milik User
exports.getMySantri = async (req, res) => {
    try {
        const [mySantri] = await pool.query('SELECT santri_id, nama_santri, nisn, satuan_pendidikan FROM santri WHERE user_id = ?', [req.user.id]);
        res.status(200).json(mySantri);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

// FUNGSI Mendapatkan Daftar Santri yang Bisa Dijemput
exports.getSantriForPickup = async (req, res) => {
    const { jadwal_id } = req.params;
    const user_id = req.user.id;
    try {
        const queryMySantri = 'SELECT santri_id, nama_santri, hubungan_wali AS status_jemput FROM santri WHERE user_id = ?';
        const querySantriTitipan = `SELECT s.santri_id, s.nama_santri, "Santri Titipan" AS status_jemput FROM izin_penjemputan ip JOIN santri s ON ip.santri_id = s.santri_id WHERE ip.user_id_penjemput = ? AND ip.jadwal_id = ? AND ip.status = 'menunggu'`;
        const [mySantri] = await pool.query(queryMySantri, [user_id]);
        const [santriTitipan] = await pool.query(querySantriTitipan, [user_id, jadwal_id]);
        const allSantri = [...mySantri, ...santriTitipan];
        res.status(200).json(allSantri);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

// FUNGSI Mengambil Data Profil
exports.getProfile = async (req, res) => {
    try {
        const idPengguna = req.user.id;
        const [wali] = await pool.query(
            "SELECT id as user_id, nama_wali, username, email, no_telp, alamat FROM users WHERE id = ? AND role = 'orang_tua'", 
            [idPengguna]
        );

        if (wali.length === 0) { 
            return res.status(404).json({ message: 'User tidak ditemukan.' }); 
        }

        const [santri] = await pool.query(
            'SELECT santri_id, nama_santri, nisn, satuan_pendidikan, hubungan_wali FROM santri WHERE user_id = ?', 
            [idPengguna]
        );

        const profileData = { ...wali[0], santri: santri };
        res.status(200).json(profileData);

    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};

// FUNGSI Memperbarui Data Profil 
exports.updateProfile = async (req, res) => {
    const userId = req.user.id;
    const { nama_wali, no_telp, alamat, santri } = req.body;
    if (!nama_wali || !no_telp || !alamat) { return res.status(400).json({ message: 'Nama, nomor telepon, dan alamat wajib diisi.' }); }
    
    const connection = await pool.getConnection();
    try {
        await connection.beginTransaction();

        // Update data wali
        await connection.query('UPDATE users SET nama_wali = ?, no_telp = ?, alamat = ? WHERE id = ?', [nama_wali, no_telp, alamat, userId]);
        // Ambil semua ID santri yang ada di database untuk user ini
        const [existingSantriRows] = await connection.query('SELECT santri_id FROM santri WHERE user_id = ?', [userId]);
        const existingSantriIds = existingSantriRows.map(s => s.santri_id);
        // Siapkan daftar ID santri yang datang dari frontend
        const incomingSantriIds = [];

        // Proses setiap santri dari frontend
        if (santri && Array.isArray(santri)) {
            for (const s of santri) {
                // Validasi input (tidak berubah)
                if (!s.nama_santri || !s.nisn || !s.satuan_pendidikan || !s.hubungan_wali) { throw new Error('Data santri tidak lengkap. Semua kolom wajib diisi.'); }
                const nisnRegex = /^\d{10}$/;
                if (!nisnRegex.test(s.nisn)) {
                    throw new Error(`NISN "${s.nisn}" tidak valid. Harus terdiri dari 10 angka.`);
                }
                
                if (s.santri_id) {
                    // JIKA ADA santri_id, berarti ini data LAMA -> lakukan UPDATE
                    incomingSantriIds.push(s.santri_id);
                    await connection.query(
                        'UPDATE santri SET nama_santri = ?, nisn = ?, satuan_pendidikan = ?, hubungan_wali = ? WHERE santri_id = ? AND user_id = ?',
                        [s.nama_santri, s.nisn, s.satuan_pendidikan, s.hubungan_wali, s.santri_id, userId]
                    );
                } else {
                    // JIKA TIDAK ADA santri_id, berarti ini data BARU -> lakukan INSERT
                    await connection.query(
                        'INSERT INTO santri (user_id, nama_santri, nisn, satuan_pendidikan, hubungan_wali) VALUES (?, ?, ?, ?, ?)',
                        [userId, s.nama_santri, s.nisn, s.satuan_pendidikan, s.hubungan_wali]
                    );
                }
            }
        }

        // Cari santri yang harus dihapus, bandingkan ID yang ada di DB dengan ID yang dikirim dari frontend
        const santriIdsToDelete = existingSantriIds.filter(id => !incomingSantriIds.includes(id));

        if (santriIdsToDelete.length > 0) {
            await connection.query('DELETE FROM santri WHERE santri_id IN (?) AND user_id = ?', [santriIdsToDelete, userId]);
        }

        await connection.commit();
        res.status(200).json({ message: 'Profil berhasil diperbarui!' });

    } catch (error) {
        await connection.rollback();
        console.error(error.message); 
        res.status(400).json({ message: error.message }); 
    } finally {
        connection.release();
    }
};

// FUNGSI Mengambil Riwayat Penjemputan
exports.getHistory = async (req, res) => {
    try {
        const userId = req.user.id;
        const [history] = await pool.query(
            `SELECT ps.pilihan_id, ps.jenis_kendaraan, ps.tgl_pilihan, jp.tgl_jemput, jp.jenis_jadwal, ss.waktu_mulai, ss.waktu_selesai, (SELECT GROUP_CONCAT(s.nama_santri SEPARATOR ', ') FROM detail_penjemputan dp JOIN santri s ON dp.santri_id = s.santri_id WHERE dp.pilihan_id = ps.pilihan_id) AS daftar_santri FROM pilihan_sesi ps JOIN sesi_penjemputan ss ON ps.sesi_id = ss.sesi_id JOIN jadwal_penjemputan jp ON ss.jadwal_id = jp.jadwal_id WHERE ps.user_id = ? ORDER BY jp.tgl_jemput DESC, ss.waktu_mulai DESC`,
            [userId]
        );
        res.status(200).json(history);
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Gagal mengambil riwayat penjemputan.' });
    }
};

// FUNGSI Mengubah Password
exports.changePassword = async (req, res) => {
    const { currentPassword, newPassword, confirmNewPassword } = req.body;
    const userId = req.user.id;

    if (!currentPassword || !newPassword || !confirmNewPassword || newPassword !== confirmNewPassword) {
        return res.status(400).json({ message: 'Input tidak valid.' });
    }
    if (newPassword.length < 8) {
        return res.status(400).json({ message: 'Password baru minimal harus 8 karakter.' });
    }

    try {
        const [userArr] = await pool.query('SELECT password FROM users WHERE id = ?', [userId]);
        if (userArr.length === 0) {
            return res.status(404).json({ message: 'User tidak ditemukan.' });
        }
        const user = userArr[0];

        const isMatch = await bcrypt.compare(currentPassword, user.password);
        if (!isMatch) {
            return res.status(401).json({ message: 'Password lama Anda salah.' });
        }

        const salt = await bcrypt.genSalt(10);
        const hashedNewPassword = await bcrypt.hash(newPassword, salt);

        await pool.query('UPDATE users SET password = ? WHERE id = ?', [hashedNewPassword, userId]);

        res.status(200).json({ message: 'Password berhasil diubah!' });
    } catch (error) {
        console.error(error);
        res.status(500).json({ message: 'Terjadi kesalahan pada server.' });
    }
};