document.addEventListener('DOMContentLoaded', () => {
    if (typeof dayjs === 'undefined') {
        console.error('Day.js library not loaded!');
        return;
    }
    dayjs.locale('id');
    const jadwalPenjemputanList = document.getElementById('jadwal-penjemputan-list');
    const jadwalKembaliList = document.getElementById('jadwal-kembali-list');
    const totalDijadwalkanElem = document.getElementById('total-dijadwalkan');
    const totalTerambilElem = document.getElementById('total-terambil');
    const sesiTableBody = document.getElementById('sesi-table-body');
    const sesiTitle = document.getElementById('sesi-title');
    const resetViewButton = document.getElementById('reset-view-button');
    
    let allDashboardData = {};

    const showPendaftarDetails = async (sesiId, waktu, tanggal) => {
        sesiTitle.textContent = `Detail Pendaftar Sesi (${waktu} - ${tanggal})`;
        sesiTableBody.innerHTML = `<tr><td colspan="6" class="p-4 text-center text-gray-400">Memuat data pendaftar...</td></tr>`;

        try {
            // Ambil data dari backend
            const pendaftarList = await fetchWithAuth(`/api/admin/sesi/${sesiId}/detail`);
            const tableHead = sesiTableBody.parentElement.querySelector('thead tr');
            tableHead.innerHTML = `
                <th class="p-3 text-left">Nama Wali</th>
                <th class="p-3 text-left">Daftar Santri</th>
                <th class="p-3 text-center">Jumlah</th>
                <th class="p-3 text-center">Kendaraan</th>
                <th class="p-3 text-center">Jalur</th>
            `;

            if (pendaftarList.length === 0) {
                sesiTableBody.innerHTML = '<tr><td colspan="5" class="text-center p-4">Belum ada pendaftar untuk sesi ini.</td></tr>';
                return;
            }
            const tableHtml = pendaftarList.map(pendaftar => {
                let jalurText = '-';
                let jalurClass = 'bg-gray-100 text-gray-800'; 

                if (pendaftar.jalur_digunakan === 'prioritas') {
                    jalurText = 'Prioritas';
                    jalurClass = 'bg-blue-100 text-blue-800';
                } else if (pendaftar.jalur_digunakan === 'reguler') {
                    jalurText = 'Reguler';
                }

                return `
                    <tr class="border-b">
                        <td class="p-3 font-medium text-gray-800">${pendaftar.nama_wali}</td>
                        <td class="p-3 text-gray-600">${pendaftar.daftar_santri}</td>
                        <td class="p-3 text-center font-medium">${pendaftar.jumlah_santri}</td>
                        <td class="p-3 text-center capitalize">${pendaftar.jenis_kendaraan}</td>
                        <td class="p-3 text-center">
                            <span class="px-2 py-1 text-xs font-semibold rounded-full ${jalurClass} capitalize">${jalurText}</span>
                        </td>
                    </tr>
                `;
            }).join('');

            sesiTableBody.innerHTML = tableHtml;

        } catch (error) {
            console.error('Error memuat detail pendaftar:', error.message);
            if (error.message !== 'Sesi berakhir') {
                sesiTableBody.innerHTML = '<tr><td colspan="5" class="text-center p-4 text-red-500">Gagal memuat data pendaftar.</td></tr>';
            }
        }
    };
    
    const showOverallSummary = () => {
        totalDijadwalkanElem.textContent = 0;
        totalTerambilElem.textContent = 0;
        const tableHead = sesiTableBody.parentElement.querySelector('thead tr');
        tableHead.innerHTML = `<th class="p-3">Sesi</th> <th class="p-3">Waktu</th> <th class="p-3">Kapasitas Santri</th> <th class="p-3">Reguler</th> <th class="p-3">Prioritas</th> <th class="p-3">Status</th>`;
        sesiTableBody.innerHTML = `<tr><td colspan="6" class="p-4 text-center text-gray-400">Klik salah satu jadwal di atas untuk melihat detail sesi.</td></tr>`;
        sesiTitle.textContent = "Detail Sesi";
        resetViewButton.classList.add('hidden');
        document.querySelectorAll('.jadwal-card').forEach(card => card.classList.remove('ring-2', 'ring-green-500', 'bg-green-50'));
    };

    const showJadwalDetails = (jadwalId) => {
        const selectedJadwal = allDashboardData.jadwal_terbaru.find(j => j.jadwal_id == jadwalId);
        if (!selectedJadwal) return;

        totalDijadwalkanElem.textContent = selectedJadwal.total_santri_dijadwalkan;
        totalTerambilElem.textContent = selectedJadwal.jumlah_santri_terdaftar;
        const jenisJadwalText = selectedJadwal.jenis_jadwal === 'kembali' ? 'Kembali' : 'Penjemputan';
        sesiTitle.textContent = `Detail Sesi ${jenisJadwalText} - ${dayjs(selectedJadwal.tgl_jemput).format('DD MMMM YYYY')}`;
        
        sesiTableBody.innerHTML = '';

        if (!selectedJadwal.sesi || selectedJadwal.sesi.length === 0) {
            sesiTableBody.innerHTML = `<tr><td colspan="6" class="p-4 text-center text-gray-400">Tidak ada sesi untuk jadwal ini.</td></tr>`;
        } else {
            selectedJadwal.sesi.forEach((sesi, index) => {
                const poinTerpakaiTotal = sesi.poin_terpakai || 0;
                const poinTerpakaiPrioritas = sesi.poin_prioritas_terpakai || 0;
                const poinTerpakaiReguler = poinTerpakaiTotal - poinTerpakaiPrioritas;
                
                const kapasitasReguler = sesi.kapasitas_poin_total - sesi.kapasitas_poin_prioritas;

                const isPenuh = poinTerpakaiTotal >= sesi.kapasitas_poin_total;
                const statusClass = isPenuh ? 'bg-red-100 text-red-700' : 'bg-green-100 text-green-700';
                const statusText = isPenuh ? 'Penuh' : 'Tersedia';
                
                const waktuMulai = sesi.waktu_mulai.substring(0,5);
                const waktuSelesai = sesi.waktu_selesai.substring(0,5);

                const row = `
                    <tr class="sesi-row border-b hover:bg-gray-50 cursor-pointer"
                        data-sesi-id="${sesi.sesi_id}"
                        data-waktu="${waktuMulai} - ${waktuSelesai}"
                        data-tanggal="${dayjs(selectedJadwal.tgl_jemput).format('DD MMMM YYYY')}">
                        <td class="p-3 font-medium">${sesi.nomor_sesi}</td>
                        <td class="p-3">${waktuMulai} - ${waktuSelesai}</td>
                        <td class="p-3">${poinTerpakaiTotal} / ${sesi.kapasitas_poin_total}</td>
                        <td class="p-3">${poinTerpakaiReguler} / ${kapasitasReguler}</td>
                        <td class="p-3">${poinTerpakaiPrioritas} / ${sesi.kapasitas_poin_prioritas}</td>
                        <td class="p-3"><span class="px-2 py-1 text-xs font-semibold rounded-full ${statusClass}">${statusText}</span></td>
                    </tr>`;
                sesiTableBody.innerHTML += row;
            });
        }
        
        resetViewButton.classList.remove('hidden');
        document.querySelectorAll('.jadwal-card').forEach(card => card.classList.remove('ring-2', 'ring-green-500', 'bg-green-50'));
        document.querySelector(`.jadwal-card[data-id='${jadwalId}']`).classList.add('ring-2', 'ring-green-500', 'bg-green-50');
    };

    const createJadwalCard = (jadwal, listElement) => {
        const formattedDate = dayjs(jadwal.tgl_jemput).format('dddd, DD MMMM YYYY');
        const card = document.createElement('div');
        card.className = 'jadwal-card p-3 rounded-lg cursor-pointer hover:bg-gray-100 transition-colors';
        card.dataset.id = jadwal.jadwal_id;
        card.innerHTML = `<p class="font-semibold text-gray-800">${formattedDate}</p><p class="text-gray-500 text-sm">${jadwal.satuan_pendidikan} - ${jadwal.waktu_mulai.substring(0,5)} - ${jadwal.waktu_selesai.substring(0,5)} WITA</p>`;
        card.addEventListener('click', () => showJadwalDetails(jadwal.jadwal_id));
        listElement.appendChild(card);
    };

    const fetchAndRender = async () => {
        try {
            allDashboardData = await fetchWithAuth('/api/admin/dashboard');
            
            jadwalPenjemputanList.innerHTML = '';
            jadwalKembaliList.innerHTML = '';

            const penjemputanJadwals = allDashboardData.jadwal_terbaru.filter(j => j.jenis_jadwal === 'penjemputan');
            const kembaliJadwals = allDashboardData.jadwal_terbaru.filter(j => j.jenis_jadwal === 'kembali');

            if(penjemputanJadwals.length > 0) penjemputanJadwals.forEach(j => createJadwalCard(j, jadwalPenjemputanList));
            else jadwalPenjemputanList.innerHTML = `<p class="text-center text-gray-400 p-3">Tidak ada jadwal penjemputan.</p>`;

            if(kembaliJadwals.length > 0) kembaliJadwals.forEach(j => createJadwalCard(j, jadwalKembaliList));
            else jadwalKembaliList.innerHTML = `<p class="text-center text-gray-400 p-3">Tidak ada jadwal kembali.</p>`;

            showOverallSummary();

        } catch (error) {
            console.error('Gagal memuat data dashboard:', error.message);
        }
    };
    
    if(resetViewButton) resetViewButton.addEventListener('click', showOverallSummary);

    sesiTableBody.addEventListener('click', (event) => {
        const row = event.target.closest('.sesi-row');
        if (row) {
            const sesiId = row.dataset.sesiId;
            const waktu = row.dataset.waktu;
            const tanggal = row.dataset.tanggal;
            showPendaftarDetails(sesiId, waktu, tanggal);
        }
    });

    fetchAndRender();
});
