document.addEventListener('DOMContentLoaded', () => {
    // Elemen DOM
    const headerTitle = document.getElementById('header-title');
    const sesiContainer = document.getElementById('sesi-container');
    const izinButton = document.getElementById('izin-button');
    const konfirmasiModal = document.getElementById('konfirmasi-modal');
    const konfirmasiForm = document.getElementById('konfirmasi-form');
    const santriListContainer = document.getElementById('santri-list-container');
    const closeKonfirmasiModalBtn = document.getElementById('close-konfirmasi-modal');
    const batalKonfirmasiBtn = document.getElementById('batal-konfirmasi');
    // Menambahkan elemen untuk notifikasi kustom (sukses/gagal)
    const notificationModal = document.getElementById('notification-modal');
    const notificationIcon = document.getElementById('notification-icon');
    const notificationTitle = document.getElementById('notification-title');
    const notificationMessage = document.getElementById('notification-message');
    const closeNotificationButton = document.getElementById('close-notification-button');

    const urlParams = new URLSearchParams(window.location.search);
    const jadwalId = urlParams.get('id');

    if (!jadwalId) {
        sesiContainer.innerHTML = '<p class="text-center text-red-500">ID Jadwal tidak ditemukan di URL.</p>';
        return;
    }

    izinButton.href = `/user-beri-izin.html?jadwal_id=${jadwalId}`;
    let selectedSesiId = null;
    let jadwalDataGlobal = null; // Simpan data jadwal secara global

    //FUNGSI UNTUK NOTIFIKASI KUSTOM ##
    const showNotification = (type, title, message) => {
        if (!notificationModal) return;
        notificationIcon.classList.remove('bg-green-500', 'bg-red-500');
        if (type === 'success') {
            notificationIcon.innerHTML = `<svg class="w-12 h-12 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>`;
            notificationIcon.classList.add('bg-green-500');
            notificationTitle.textContent = title || 'Berhasil!';
        } else { 
            notificationIcon.innerHTML = `<svg class="w-12 h-12 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>`;
            notificationIcon.classList.add('bg-red-500');
            notificationTitle.textContent = title || 'Gagal!';
        }
        notificationMessage.textContent = message;
        notificationModal.classList.remove('hidden');
    };

    const closeNotification = () => {
        if (notificationModal) notificationModal.classList.add('hidden');
        loadSesiData(); // Muat ulang data setelah notifikasi ditutup
    };

    if (closeNotificationButton) {
        closeNotificationButton.addEventListener('click', closeNotification);
    }
    
    async function loadSesiData() {
        try {
            jadwalDataGlobal = await fetchWithAuth(`/api/jadwal/${jadwalId}`);
            renderHeader(jadwalDataGlobal);
            renderSesiCards(jadwalDataGlobal.sesi);
        } catch (error) {
            console.error(error.message);
            if (error.message !== 'Sesi berakhir') {
                sesiContainer.innerHTML = `<p class="text-center text-red-500">${error.message}</p>`;
            }
        }
    }
    
    function renderHeader(jadwalData) {
        headerTitle.textContent = `Jadwal ${jadwalData.satuan_pendidikan}`;
    }

    function renderSesiCards(sesiList) {
        if (!sesiList || sesiList.length === 0) {
            sesiContainer.innerHTML = '<p class="text-center text-gray-500">Tidak ada sesi tersedia untuk jadwal ini.</p>';
            return;
        }
        sesiContainer.innerHTML = sesiList.map(createSesiCard).join('');
        document.querySelectorAll('.pilih-sesi-btn').forEach(button => {
            button.addEventListener('click', (event) => {
                selectedSesiId = event.currentTarget.dataset.sesiId;
                openKonfirmasiModal();
            });
        });
    }

    // FUNGSI createSesiCard DENGAN PESAN INFO
    function createSesiCard(sesi) {
        const waktuMulai = dayjs(sesi.waktu_mulai, 'HH:mm:ss').format('HH:mm');
        const waktuSelesai = dayjs(sesi.waktu_selesai, 'HH:mm:ss').format('HH:mm');
        const isPenuh = sesi.status_ketersediaan === 'Penuh';

        const kapasitasText = `Kapasitas: ${sesi.poin_terpakai} / ${sesi.kapasitas_poin_total} Santri`;

        return `
            <div class="bg-white p-4 rounded-xl shadow-sm border">
                <div class="flex justify-between items-start">
                    <div class="flex-1 pr-4">
                        <h3 class="text-2xl font-bold text-green-600">SESI ${sesi.nomor_sesi}</h3>
                        <p class="text-gray-600">${waktuMulai} - ${waktuSelesai} WITA</p>
                        <div class="flex items-center space-x-2 mt-2 text-gray-700">
                             <svg class="w-5 h-5 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283-.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path></svg>
                             <span class="text-sm">${kapasitasText}</span>
                        </div>
                    </div>
                    <button data-sesi-id="${sesi.sesi_id}" class="pilih-sesi-btn text-white font-semibold py-2 px-6 rounded-lg flex-shrink-0 ${isPenuh ? 'bg-red-500 cursor-not-allowed' : 'bg-green-600 hover:bg-green-700'}" ${isPenuh ? 'disabled' : ''}>
                        ${isPenuh ? 'Penuh' : 'Pilih'}
                    </button>
                </div>
            </div>
        `;
    }
    
    async function openKonfirmasiModal() {
        try {
            santriListContainer.innerHTML = '<p class="text-center text-gray-500">Memuat data santri...</p>';
            konfirmasiModal.classList.remove('hidden');
            const santriForPickup = await fetchWithAuth(`/api/user/santri-jemputan/${jadwalId}`);
            if (santriForPickup.length > 0) {
                santriListContainer.innerHTML = santriForPickup.map(santri => `
                    <label class="p-3 border rounded-lg flex items-center cursor-pointer hover:bg-gray-50">
                        <input type="checkbox" name="santri_ids" value="${santri.santri_id}" class="mr-3 h-5 w-5">
                        <span class="font-medium text-gray-800">${santri.nama_santri}</span>
                        <span class="ml-auto text-xs font-semibold ${santri.status_jemput === 'Santri Titipan' ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800'} px-2 py-1 rounded-full">${santri.status_jemput}</span>
                    </label>
                `).join('');
            } else {
                santriListContainer.innerHTML = '<p class="text-center text-gray-500">Anda tidak memiliki data santri yang bisa dijemput untuk jadwal ini.</p>';
            }
        } catch (error) {
            console.error(error.message);
            if (error.message !== 'Sesi berakhir') {
                // Mengganti alert dengan notifikasi kustom
                showNotification('error', 'Gagal', 'Gagal memuat daftar santri. Silakan tutup dan coba lagi.');
                closeKonfirmasiModal();
            }
        }
    }
    
    function closeKonfirmasiModal() {
        konfirmasiModal.classList.add('hidden');
        konfirmasiForm.reset();
    }
    // Fungsi handleKonfirmasiSubmit dengan notifikasi
    async function handleKonfirmasiSubmit(event) {
        event.preventDefault();
        const formData = new FormData(konfirmasiForm);
        const selectedSantriIds = formData.getAll('santri_ids');
        const jenisKendaraan = formData.get('jenis_kendaraan');

        if (selectedSantriIds.length === 0) {
            // Menggunakan notifikasi kustom, bukan alert()
            showNotification('error', 'Peringatan', 'Silakan pilih minimal satu santri untuk dijemput.');
            return;
        }

        try {
            const result = await fetchWithAuth('/api/jadwal/pilih-sesi', {
                method: 'POST',
                body: JSON.stringify({
                    sesi_id: selectedSesiId,
                    santri_ids: selectedSantriIds,
                    jenis_kendaraan: jenisKendaraan
                })
            });
            
            closeKonfirmasiModal();
            // Tampilkan notifikasi sukses kustom
            showNotification('success', 'Berhasil!', result.message);

        } catch (error) {
            console.error(error.message);
            if (error.message !== 'Sesi berakhir') {
                // Tampilkan pesan error spesifik dari backend di notifikasi kustom
                showNotification('error', 'Gagal Mendaftar', error.message);
            }
        }
    }
    
    closeKonfirmasiModalBtn.addEventListener('click', closeKonfirmasiModal);
    batalKonfirmasiBtn.addEventListener('click', closeKonfirmasiModal);
    konfirmasiForm.addEventListener('submit', handleKonfirmasiSubmit);

    loadSesiData();
});

