const { pool } = require('../config/database');
const webpush = require('web-push');

/**
 * Mengirim notifikasi ke semua pengguna yang terdaftar.
 * @param {object} payload - Objek berisi title dan body notifikasi.
 */
exports.sendNotificationToAll = async (payload) => {
    try {
        // Ambil semua data langganan dari database
        const [subscriptions] = await pool.query('SELECT * FROM push_subscriptions');
        
        if (subscriptions.length === 0) {
            console.log('Tidak ada pengguna yang terdaftar untuk notifikasi.');
            return;
        }

        console.log(`Mengirim notifikasi ke ${subscriptions.length} pengguna...`);
        const payloadString = JSON.stringify(payload);

        // Kirim notifikasi ke setiap pengguna
        const promises = subscriptions.map(sub => {
            const subscription = {
                endpoint: sub.endpoint,
                keys: {
                    p256dh: sub.p256dh,
                    auth: sub.auth
                }
            };
            
            return webpush.sendNotification(subscription, payloadString)
                .catch(error => {
                    // Jika subscription sudah tidak valid (misal: user ganti browser/uninstall)
                    // Error code 410 menandakan "Gone" atau sudah tidak ada.
                    if (error.statusCode === 410) {
                        console.log('Subscription sudah tidak berlaku, menghapus dari DB:', sub.id);
                        // Hapus subscription yang tidak valid dari database
                        return pool.query('DELETE FROM push_subscriptions WHERE id = ?', [sub.id]);
                    } else {
                        console.error('Gagal mengirim notifikasi ke sub ID:', sub.id, error.body);
                    }
                });
        });

        // Tunggu semua proses pengiriman selesai
        await Promise.all(promises);
        console.log('Semua notifikasi berhasil dikirim (atau yang gagal sudah ditangani).');

    } catch (error) {
        console.error('Error besar saat proses pengiriman notifikasi:', error);
    }
};