// services/reminderService.js

const cron = require('node-cron');
const { pool } = require('../config/database');
const webpush = require('web-push');
const dayjs = require('dayjs');

async function sendNotification(subscription, payload) {
    const subscriptionObject = {
        endpoint: subscription.endpoint,
        keys: { p256dh: subscription.p256dh, auth: subscription.auth }
    };
    try {
        await webpush.sendNotification(subscriptionObject, JSON.stringify(payload));
    } catch (error) {
        if (error.statusCode === 410 || error.statusCode === 404) {
            console.log('Subscription tidak berlaku, menghapus dari DB (ID):', subscription.id);
            await pool.query('DELETE FROM push_subscriptions WHERE id = ?', [subscription.id]);
        } else {
            console.error('Gagal mengirim notifikasi (ID Sub: ' + subscription.id + '):', error.body);
        }
    }
}

async function checkSchedulesAndSendReminders() {
    console.log('⏰ Menjalankan pengecekan jadwal (Logika Final)...');
    
    try {
        // ## LOGIKA WAKTU FINAL DENGAN TIMESTAMP() ##
        // Menggunakan TIMESTAMP() adalah cara yang paling benar dan andal untuk menggabungkan DATE dan TIME.
        const [schedules] = await pool.query(`
            SELECT DISTINCT
                j.jadwal_id, j.jenis_jadwal, j.satuan_pendidikan,
                TIMESTAMP(j.tgl_jemput, j.waktu_mulai) AS waktu_penjemputan,
                ps.id AS subscription_id, ps.endpoint, ps.p256dh, ps.auth
            FROM jadwal_penjemputan j
            JOIN santri s ON j.satuan_pendidikan = s.satuan_pendidikan
            JOIN push_subscriptions ps ON s.user_id = ps.user_id
            WHERE 
                -- Cek untuk pengingat 24 jam
                (
                    TIMESTAMP(j.tgl_jemput, j.waktu_mulai) - INTERVAL 24 HOUR 
                    BETWEEN NOW() - INTERVAL 30 MINUTE AND NOW()
                ) OR
                -- Cek untuk pengingat 12 jam
                (
                    TIMESTAMP(j.tgl_jemput, j.waktu_mulai) - INTERVAL 12 HOUR 
                    BETWEEN NOW() - INTERVAL 30 MINUTE AND NOW()
                )
        `);

        if (schedules.length === 0) {
            console.log('Tidak ada jadwal yang perlu diingatkan saat ini.');
            return;
        }

        console.log(`✅ Menemukan ${schedules.length} notifikasi pengingat untuk dikirim.`);

        for (const schedule of schedules) {
            const now = new Date();
            const scheduleTime = new Date(schedule.waktu_penjemputan);
            const hoursRemaining = Math.round((scheduleTime - now) / (1000 * 60 * 60));
            const timeFrame = hoursRemaining > 18 ? "1 hari lagi" : "sekitar 12 jam lagi";

            const payload = {
                title: `Pengingat Jadwal Santri ${schedule.satuan_pendidikan}`,
                body: `Jadwal ${schedule.jenis_jadwal} santri ${schedule.satuan_pendidikan} ${timeFrame}. Pastikan Anda sudah memilih sesi dan datang tepat waktu!`
            };
            const subscription = {
                id: schedule.subscription_id, endpoint: schedule.endpoint,
                p256dh: schedule.p256dh, auth: schedule.auth
            };
            await sendNotification(subscription, payload);
        }
    } catch (error) {
        console.error('Terjadi error dalam cron job pengecekan jadwal:', error);
    }
}

exports.startReminderService = () => {
    cron.schedule('*/30 * * * *', checkSchedulesAndSendReminders, {
        scheduled: true,
        timezone: "Asia/Makassar" 
    });
    console.log('✅ Layanan Pengingat Notifikasi (Cron Job) telah dimulai. Pengecekan setiap 30 menit.');
    checkSchedulesAndSendReminders();
};